/*
 * Copyright (C) 2000 - 2024 Silverpeas
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * As a special exception to the terms and conditions of version 3.0 of
 * the GPL, you may redistribute this Program in connection with Free/Libre
 * Open Source Software ("FLOSS") applications as described in Silverpeas's
 * FLOSS exception.  You should have received a copy of the text describing
 * the FLOSS exception, and it is also available here:
 * "https://www.silverpeas.org/legal/floss_exception.html"
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.silverpeas.core.webapi.util.logging;

import org.silverpeas.core.web.rs.WebEntity;
import org.silverpeas.kernel.logging.Level;
import org.silverpeas.kernel.logging.LoggerConfigurationManager.LoggerConfiguration;
import org.silverpeas.kernel.logging.SilverLoggerProvider;

import javax.xml.bind.annotation.XmlElement;
import java.net.URI;
import java.util.Arrays;

/**
 * Web entity used to carries the configuration of a given logger between a client and a REST
 * resource.
 * @author mmoquillon
 */
public class LoggerConfigurationEntity implements WebEntity {

  /**
   * Constant that defines the level of a logger is the one of one of its parent. It is defaulted
   * to one of its parent logger.
   */
  public static final String PARENT_LEVEL = "PARENT";

  @XmlElement(defaultValue = "")
  private URI uri;
  private String logger;
  private String level;

  @SuppressWarnings("unused")
  protected LoggerConfigurationEntity() {
    // for serialization from JSON
  }

  private LoggerConfigurationEntity(String logger, String level) {
    this.logger = logger;
    this.level = (level == null || level.trim().isEmpty() ? PARENT_LEVEL : level);
  }

  /**
   * Constructs from the specified logger configuration its Web entity representation ready to
   * be serialized in a given format (MIME type).
   * @param configuration a logger configuration.
   * @return the Web entity representation of the logger configuration.
   */
  public static LoggerConfigurationEntity toWebEntity(LoggerConfiguration configuration) {
    String level =
        (configuration.getLevel() == null ? PARENT_LEVEL : configuration.getLevel().name());
    return new LoggerConfigurationEntity(configuration.getNamespace(), level);
  }

  public LoggerConfigurationEntity withAsURi(URI uri) {
    this.uri = uri;
    return this;
  }

  /**
   * Gets the URI at which this web entity is published and can be accessed.
   * @return the web entity URI.
   */
  @Override
  public URI getURI() {
    return uri;
  }

  public String getLogger() {
    return logger;
  }

  public String getLevel() {
    return level;
  }

  public LoggerConfiguration toLoggerConfiguration() {
    Level loggingLevel = (isLevelDefined(getLevel()) ? Level.valueOf(getLevel()) : null);
    return SilverLoggerProvider.getInstance()
        .getConfigurationManager()
        .getLoggerConfiguration(getLogger())
        .withLevel(loggingLevel);
  }

  private boolean isLevelDefined(String level) {
    return Arrays.stream(Level.values())
        .anyMatch(l -> l.name().equals(level));
  }
}
